// simple color map for badge
const badgeColor = (n) => {
  if (n >= 80) return "#1FAA59"; // green
  if (n >= 60) return "#F4C430"; // amber
  return "#E94444";              // red
};

chrome.runtime.onInstalled.addListener(async () => {
  const defaults = {
    weights: {
      readability: 1.2,
      depth: 1.4,
      citations: 1.1,
      originality: 1.2,
      media: 0.6,
      byline: 0.5,
      freshness: 1.0,
      noise: 1.2,      // penalty
      performance: 0.6,
      headline: 0.7
    },
    lastScore: null
  };
  const existing = await chrome.storage.local.get(null);
  if (!existing.weights) chrome.storage.local.set(defaults);
});

async function scoreActiveTab() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab?.id) return;

  const [{ result: metrics }] = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: collectMetrics
  });

  const { weights } = await chrome.storage.local.get("weights");
  const scored = computeScore(metrics, weights);

  await chrome.storage.local.set({ lastScore: { url: tab.url, metrics, scored, ts: Date.now() } });
  chrome.action.setBadgeText({ tabId: tab.id, text: String(scored.total) });
  chrome.action.setBadgeBackgroundColor({ tabId: tab.id, color: badgeColor(scored.total) });
  chrome.runtime.sendMessage({ type: "SCORE_READY", payload: { metrics, scored } });
}

chrome.commands.onCommand.addListener((cmd) => { if (cmd === "score-page") scoreActiveTab(); });

chrome.runtime.onMessage.addListener((msg, _s, send) => {
  if (msg?.type === "SCORE_REQUEST") scoreActiveTab().then(() => send(true));
  return true;
});

// --------- analyzer & scorer (runs in page) ----------
function collectMetrics() {
  const textFrom = (root) => {
    const walker = document.createTreeWalker(root || document.body, NodeFilter.SHOW_TEXT, null);
    let s = "", n;
    while ((n = walker.nextNode())) s += " " + n.nodeValue;
    return s.replace(/\s+/g, " ").trim();
  };

  const readability = (() => {
    const t = textFrom(document.body);
    const sentences = (t.match(/[.!?]+/g) || []).length || 1;
    const words = (t.match(/\b[\w’'-]+\b/g) || []).length || 1;
    const syllables = (t.match(/[aeiouy]+/gi) || []).length || 1;
    const flesch = 206.835 - 1.015 * (words / sentences) - 84.6 * (syllables / words);
    return Math.max(0, Math.min(100, Math.round(((flesch + 50) / 170) * 100)));
  })();

  const depth = (() => {
    const main = document.querySelector("article, main") || document.body;
    const words = (textFrom(main).match(/\b[\w’'-]+\b/g) || []).length;
    return Math.max(0, Math.min(100, Math.round((Math.min(words, 2500) / 2500) * 100)));
  })();

  const citations = (() => {
    const links = [...document.querySelectorAll("a[href]")];
    const good = links.filter(a => {
      const href = a.getAttribute("href") || "";
      const txt = (a.textContent || "").toLowerCase();
      const refy = /(source|study|paper|doi|reference|dataset|github)/.test(txt);
      const ext = /^https?:\/\//.test(href) && !href.includes(location.host);
      return (refy && ext) || (ext && a.closest("sup, .citation, .references"));
    }).length;
    return Math.max(0, Math.min(100, Math.round((Math.min(good, 10) / 10) * 100)));
  })();

  const originality = (() => {
    const t = textFrom(document.body).toLowerCase();
    const tokens = (t.match(/\b[a-z0-9’'-]{3,}\b/g) || []);
    const uniq = new Set(tokens).size;
    const r = tokens.length ? uniq / tokens.length : 0;
    return Math.max(0, Math.min(100, Math.round(r * 120)));
  })();

  const media = (() => {
    const imgs = [...document.images].filter(i => i.width * i.height > 30000);
    const vids = [...document.querySelectorAll("video")];
    const annotated = imgs.filter(i => i.alt?.length > 3).length + vids.length;
    const score = Math.min(100, Math.round((annotated / Math.max(imgs.length + vids.length, 1)) * 100));
    return score;
  })();

  const byline = (() => {
    const author = document.querySelector('[rel="author"], .byline, [itemprop="author"], .author, .post-author');
    return author ? 100 : 30;
  })();

  const freshness = (() => {
    const cand = document.querySelector('time[datetime], time, meta[property="article:published_time"], meta[name="date"], [data-published]');
    let d;
    if (cand?.getAttribute) d = cand.getAttribute("datetime") || cand.getAttribute("content") || cand.textContent;
    const dt = d ? new Date(d) : null;
    const days = dt && !Number.isNaN(+dt) ? (Date.now() - dt.getTime()) / 86400000 : 365;
    const s = Math.max(10, Math.min(100, Math.round(100 - Math.min(days, 365) / 365 * 90)));
    return s;
  })();

  const noise = (() => {
    const scripts = [...document.scripts].map(s => s.src).join(" ");
    const iframes = [...document.querySelectorAll("iframe")].map(i => i.src).join(" ");
    const blob = (scripts + " " + iframes).toLowerCase();
    const bad = [
      "doubleclick", "googletag", "adsystem", "advert", "pixel", "beacon",
      "hotjar", "mixpanel", "segment", "outbrain", "taboola"
    ].filter(k => blob.includes(k)).length;
    return Math.max(0, Math.min(100, 100 - bad * 12));
  })();

  const performance = (() => {
    const nodes = document.getElementsByTagName("*").length;
    const scripts = document.scripts.length;
    const domScore = Math.max(0, 100 - Math.min(nodes / 3000, 1) * 70);
    const jsScore = Math.max(0, 100 - Math.min(scripts / 60, 1) * 30);
    return Math.round((domScore + jsScore) / 2);
  })();

  const headline = (() => {
    const title = document.title || "";
    const bad = /(you won't believe|shocking|#\d+ things|this one trick|blow your mind)/i.test(title);
    const q = /\?/.test(title);
    const len = title.length;
    let score = 80;
    if (bad) score -= 40;
    if (q) score -= 10;
    if (len < 25 || len > 90) score -= 10;
    return Math.max(0, Math.min(100, score));
  })();

  return { readability, depth, citations, originality, media, byline, freshness, noise, performance, headline };
}

function computeScore(m, w) {
  const keys = Object.keys(w);
  const sumW = keys.reduce((a, k) => a + Math.abs(w[k] || 0), 0) || 1;
  const norm = Object.fromEntries(keys.map(k => [k, (w[k] || 0) / sumW]));

  const parts = {
    readability: m.readability * norm.readability,
    depth: m.depth * norm.depth,
    citations: m.citations * norm.citations,
    originality: m.originality * norm.originality,
    media: m.media * norm.media,
    byline: m.byline * norm.byline,
    freshness: m.freshness * norm.freshness,
    performance: m.performance * norm.performance,
    headline: m.headline * norm.headline,
    noise: (100 - m.noise) * norm.noise // penalty
  };

  const positive = parts.readability + parts.depth + parts.citations + parts.originality +
                   parts.media + parts.byline + parts.freshness + parts.performance + parts.headline;
  const penalty = parts.noise;
  const total = Math.max(0, Math.min(100, Math.round(positive - penalty)));

  return {
    total,
    breakdown: {
      readability: m.readability,
      depth: m.depth,
      citations: m.citations,
      originality: m.originality,
      media: m.media,
      byline: m.byline,
      freshness: m.freshness,
      performance: m.performance,
      headline: m.headline,
      noisePenalty: 100 - m.noise
    },
    weights: norm
  };
}
